import pygame
import math

# Initialize Pygame
pygame.init()

# Screen dimensions
WIDTH, HEIGHT = 800, 600
screen = pygame.display.set_mode((WIDTH, HEIGHT))
pygame.display.set_caption("Earth and Moon Game")

# Colors
BLACK = (0, 0, 0)
WHITE = (255, 255, 255)

# Load images
earth_img = pygame.image.load('earth.png').convert_alpha()
moon_img = pygame.image.load('moon.png').convert_alpha()
ship_img = pygame.image.load('spaceship.png').convert_alpha()

# Resize images
EARTH_RADIUS = 100
MOON_RADIUS = 30
SHIP_SIZE = (50, 50)

earth_img = pygame.transform.scale(earth_img, (2 * EARTH_RADIUS, 2 * EARTH_RADIUS))
moon_img = pygame.transform.scale(moon_img, (2 * MOON_RADIUS, 2 * MOON_RADIUS))
ship_img = pygame.transform.scale(ship_img, SHIP_SIZE)

# Clock to control the frame rate
clock = pygame.time.Clock()

# Earth position
earth_pos = (WIDTH // 2, HEIGHT // 2)

# Moon orbit parameters
moon_orbit_radius = 150
MOON_SPEED = 0.01

# Ship variables
ship_pos = [WIDTH // 4, HEIGHT // 2]
ship_vel = [0, 0]
SHIP_SPEED = 2

# Laser properties
LASER_SPEED = 5
LASER_SIZE = (5, 10)
LASER_COLOR = WHITE

class Laser:
    def __init__(self, x, y, angle):
        self.x = x
        self.y = y
        self.angle = angle
        self.active = True

    def update(self):
        # Update laser position based on its angle and speed
        self.x += LASER_SPEED * math.sin(math.radians(self.angle))
        self.y -= LASER_SPEED * math.cos(math.radians(self.angle))

        # Deactivate the laser if it goes off-screen
        if (self.x < 0 or self.x > WIDTH or
            self.y < 0 or self.y > HEIGHT):
            self.active = False

    def draw(self, screen):
        # Draw the laser as a rectangle
        end_x = self.x + LASER_SIZE[0] * math.sin(math.radians(self.angle))
        end_y = self.y - LASER_SIZE[0] * math.cos(math.radians(self.angle))
        pygame.draw.line(screen, LASER_COLOR, (self.x, self.y), (end_x, end_y), LASER_SIZE[1])

def rotate_image(image, angle):
    """Rotate an image while keeping its center."""
    orig_rect = image.get_rect()
    rot_image = pygame.transform.rotate(image, angle)
    rot_rect = orig_rect.copy()
    rot_rect.center = rot_image.get_rect().center
    rot_image = rot_image.subsurface(rot_rect).copy()
    return rot_image

def main():
    running = True
    lasers = []
    moon_angle = 0
    ship_angle = 0

    while running:
        for event in pygame.event.get():
            if event.type == pygame.QUIT:
                running = False
            elif event.type == pygame.KEYDOWN:
                if event.key == pygame.K_SPACE:
                    # Create a new laser at the ship's position and angle
                    new_laser = Laser(ship_pos[0], ship_pos[1], ship_angle)
                    lasers.append(new_laser)

        keys = pygame.key.get_pressed()

        # Move the ship based on arrow keys
        if keys[pygame.K_UP]:
            ship_pos[1] -= SHIP_SPEED * math.cos(math.radians(ship_angle))
            ship_pos[0] += SHIP_SPEED * math.sin(math.radians(ship_angle))
        if keys[pygame.K_DOWN]:
            ship_pos[1] += SHIP_SPEED * math.cos(math.radians(ship_angle))
            ship_pos[0] -= SHIP_SPEED * math.sin(math.radians(ship_angle))
        if keys[pygame.K_LEFT]:
            ship_angle -= 5
        if keys[pygame.K_RIGHT]:
            ship_angle += 5

        # Update laser positions and remove inactive lasers
        for laser in lasers:
            laser.update()
        lasers = [laser for laser in lasers if laser.active]

        # Update moon position
        moon_x = earth_pos[0] + moon_orbit_radius * math.cos(moon_angle)
        moon_y = earth_pos[1] - moon_orbit_radius * math.sin(moon_angle)
        moon_angle += MOON_SPEED

        # Clear the screen
        screen.fill(BLACK)

        # Draw the Earth, Moon, Ship, and Lasers
        screen.blit(earth_img, (earth_pos[0] - EARTH_RADIUS, earth_pos[1] - EARTH_RADIUS))
        screen.blit(moon_img, (moon_x - MOON_RADIUS, moon_y - MOON_RADIUS))
        rotated_ship = rotate_image(ship_img, -ship_angle)
        ship_rect = rotated_ship.get_rect(center=(ship_pos[0], ship_pos[1]))
        screen.blit(rotated_ship, ship_rect.topleft)

        for laser in lasers:
            laser.draw(screen)

        # Update the display
        pygame.display.flip()

        # Cap the frame rate
        clock.tick(60)

    pygame.quit()

if __name__ == "__main__":
    main()